<?php
/**
 * @title          User Core Class
 *
 * @author         Pierre-Henry Soria <ph7software@gmail.com>
 * @copyright      (c) 2012-2017, Pierre-Henry Soria. All Rights Reserved.
 * @license        GNU General Public License; See PH7.LICENSE.txt and PH7.COPYRIGHT.txt in the root directory.
 * @package        PH7 / App / System / Core / Class
 */

namespace PH7;

use PH7\Framework\Cache\Cache;
use PH7\Framework\Config\Config;
use PH7\Framework\File\File;
use PH7\Framework\Image\Image;
use PH7\Framework\Ip\Ip;
use PH7\Framework\Layout\Html\Design;
use PH7\Framework\Mvc\Model\DbConfig;
use PH7\Framework\Mvc\Model\Engine\Util\Various as VariousModel;
use PH7\Framework\Mvc\Model\Security as SecurityModel;
use PH7\Framework\Mvc\Request\Http as HttpRequest;
use PH7\Framework\Mvc\Router\Uri;
use PH7\Framework\Navigation\Browser;
use PH7\Framework\Registry\Registry;
use PH7\Framework\Security\Validate\Validate;
use PH7\Framework\Session\Session;
use PH7\Framework\Str\Str;
use PH7\Framework\Url\Header;
use PH7\Framework\Url\Url;
use PH7\Framework\Util\Various;
use stdClass;

// Abstract Class
class UserCore
{
    /**
     * Users'levels.
     *
     * @return bool
     */
    public static function auth()
    {
        $oSession = new Session;
        $bIsConnected = ((int)$oSession->exists('member_id')) && $oSession->get('member_ip') === Ip::get() && $oSession->get('member_http_user_agent') === (new Browser)->getUserAgent();

        /** Destroy the object to minimize the CPU resources **/
        unset($oSession);

        return $bIsConnected;
    }

    /**
     * Check if an admin is logged as a user.
     *
     * @return bool
     */
    public static function isAdminLoggedAs()
    {
        return (new Session)->exists('login_user_as');
    }

    /**
     * Delete User.
     *
     * @param integer $iProfileId
     * @param string $sUsername
     *
     * @return void
     */
    public function delete($iProfileId, $sUsername)
    {
        if ($sUsername == PH7_GHOST_USERNAME) exit('You cannot delete this profile!');

        $oFile = new File;
        $oFile->deleteDir(PH7_PATH_PUBLIC_DATA_SYS_MOD . 'user/avatar/' . PH7_IMG . $sUsername);
        $oFile->deleteDir(PH7_PATH_PUBLIC_DATA_SYS_MOD . 'user/background/' . PH7_IMG . $sUsername);
        $oFile->deleteDir(PH7_PATH_PUBLIC_DATA_SYS_MOD . 'picture/' . PH7_IMG . $sUsername);
        $oFile->deleteDir(PH7_PATH_PUBLIC_DATA_SYS_MOD . 'video/file/' . $sUsername);
        $oFile->deleteDir(PH7_PATH_PUBLIC_DATA_SYS_MOD . 'note/' . PH7_IMG . $sUsername);
        unset($oFile);

        (new UserCoreModel)->delete($iProfileId, $sUsername);

        /* Clean UserCoreModel and Avatar Cache */
        (new Cache)->start(UserCoreModel::CACHE_GROUP, null, null)->clear()
            ->start(Design::CACHE_AVATAR_GROUP . $sUsername, null, null)->clear();
    }

    /**
     * Set the avatar file and add it to the database.
     *
     * @param integer $iProfileId
     * @param integer $sUsername
     * @param string $sFile
     * @param integer $iApproved (1 = approved 0 = pending)
     *
     * @return bool TRUE if success, FALSE if the extension is wrong.
     */
    public function setAvatar($iProfileId, $sUsername, $sFile, $iApproved = 1)
    {
        /**
         * This can cause minor errors (eg if a user sent a file that is not a photo).
         * So we hide the errors if we are not in development mode.
         */
        if (!isDebug()) error_reporting(0);

        $oAvatar1 = new Image($sFile, 600, 800);

        if (!$oAvatar1->validate()) return false; // File type incompatible.

        // We removes the old avatar if it exists and we delete the cache at the same time.
        $this->deleteAvatar($iProfileId, $sUsername);

        $oAvatar2 = clone $oAvatar1;
        $oAvatar3 = clone $oAvatar1;
        $oAvatar4 = clone $oAvatar1;
        $oAvatar5 = clone $oAvatar1;
        $oAvatar6 = clone $oAvatar1;
        $oAvatar7 = clone $oAvatar1;
        $oAvatar2->square(32);
        $oAvatar3->square(64);
        $oAvatar4->square(100);
        $oAvatar5->square(150);
        $oAvatar6->square(200);
        $oAvatar7->resize(400);

        /* Set watermark text on large avatars */
        $sWatermarkText = DbConfig::getSetting('watermarkTextImage');
        $iSizeWatermarkText = DbConfig::getSetting('sizeWatermarkTextImage');
        $oAvatar4->watermarkText($sWatermarkText, $iSizeWatermarkText);
        $oAvatar5->watermarkText($sWatermarkText, $iSizeWatermarkText);
        $oAvatar6->watermarkText($sWatermarkText, $iSizeWatermarkText);
        $oAvatar7->watermarkText($sWatermarkText, $iSizeWatermarkText);

        $sPath = PH7_PATH_PUBLIC_DATA_SYS_MOD . 'user/avatar/img/' . $sUsername . PH7_SH;
        (new File)->createDir($sPath);

        $sFileName = Various::genRnd($oAvatar1->getFileName(), 1);

        $sFile1 = $sFileName . '.' . $oAvatar1->getExt();  // Original, four characters
        $sFile2 = $sFileName . '-32.' . $oAvatar2->getExt();
        $sFile3 = $sFileName . '-64.' . $oAvatar3->getExt();
        $sFile4 = $sFileName . '-100.' . $oAvatar4->getExt();
        $sFile5 = $sFileName . '-150.' . $oAvatar5->getExt();
        $sFile6 = $sFileName . '-200.' . $oAvatar6->getExt();
        $sFile7 = $sFileName . '-400.' . $oAvatar7->getExt();

        // Add the avatar
        (new UserCoreModel)->setAvatar($iProfileId, $sFile1, $iApproved);

        /* Saved the new avatars */
        $oAvatar1->save($sPath . $sFile1);
        $oAvatar2->save($sPath . $sFile2);
        $oAvatar3->save($sPath . $sFile3);
        $oAvatar4->save($sPath . $sFile4);
        $oAvatar5->save($sPath . $sFile5);
        $oAvatar6->save($sPath . $sFile6);
        $oAvatar7->save($sPath . $sFile7);

        unset($oAvatar1, $oAvatar2, $oAvatar3, $oAvatar4, $oAvatar5, $oAvatar6, $oAvatar7);

        return true;
    }

    /**
     * Delete the avatar (image) and track database.
     *
     * @param integer $iProfileId
     * @param string $sUsername
     *
     * @return void
     */
    public function deleteAvatar($iProfileId, $sUsername)
    {
        // We start to delete the file before the data in the database if we could not delete the file since we would have lost the link to the file found in the database.
        $sGetAvatar = (new UserCoreModel)->getAvatar($iProfileId, null);
        $sFile = $sGetAvatar->pic;

        $oFile = new File;
        $sExt = PH7_DOT . $oFile->getFileExt($sFile);

        $sPath = PH7_PATH_PUBLIC_DATA_SYS_MOD . 'user/avatar/img/' . $sUsername . PH7_SH;

        /** Array to the new format (>= PHP5.4) **/
        $aFiles = [
            $sPath . $sFile,
            $sPath . str_replace($sExt, '-32' . $sExt, $sFile),
            $sPath . str_replace($sExt, '-64' . $sExt, $sFile),
            $sPath . str_replace($sExt, '-100' . $sExt, $sFile),
            $sPath . str_replace($sExt, '-150' . $sExt, $sFile),
            $sPath . str_replace($sExt, '-200' . $sExt, $sFile),
            $sPath . str_replace($sExt, '-400' . $sExt, $sFile),
        ];

        $oFile->deleteFile($aFiles);
        unset($oFile);

        (new UserCoreModel)->deleteAvatar($iProfileId);

        /* Clean User Avatar Cache */
        (new Cache)->start(Design::CACHE_AVATAR_GROUP . $sUsername, null, null)->clear()
            ->start(UserCoreModel::CACHE_GROUP, 'avatar' . $iProfileId, null)->clear();
    }

    /**
     * Set a background on user profile.
     *
     * @param integer $iProfileId
     * @param string $sUsername
     * @param string $sFile
     * @param integer $iApproved (1 = approved 0 = pending)
     *
     * @return bool TRUE if success, FALSE if the extension is wrong.
     */
    public function setBackground($iProfileId, $sUsername, $sFile, $iApproved = 1)
    {
        /**
         * This can cause minor errors (eg if a user sent a file that is not a photo).
         * So we hide the errors if we are not in development mode.
         */
        if (!isDebug()) error_reporting(0);

        $oWallpaper = new Image($sFile, 600, 800);

        if (!$oWallpaper->validate()) return false;

        // We removes the old background if it exists and we delete the cache at the same time.
        $this->deleteBackground($iProfileId, $sUsername);


        $sPath = PH7_PATH_PUBLIC_DATA_SYS_MOD . 'user/background/img/' . $sUsername . PH7_SH;
        (new File)->createDir($sPath);

        $sFileName = Various::genRnd($oWallpaper->getFileName(), 1);
        $sFile = $sFileName . '.' . $oWallpaper->getExt();

        // Add the profile background
        (new UserCoreModel)->addBackground($iProfileId, $sFile, $iApproved);

        // Saved the new background
        $oWallpaper->save($sPath . $sFile);

        unset($oWallpaper);

        return true;
    }

    /**
     * @param integer $iProfileId
     * @param string $sUsername
     *
     * @return void
     */
    public function deleteBackground($iProfileId, $sUsername)
    {
        // We start to delete the file before the data in the database if we could not delete the file since we would have lost the link to the file found in the database.
        $sFile = (new UserCoreModel)->getBackground($iProfileId, null);

        (new File)->deleteFile(PH7_PATH_PUBLIC_DATA_SYS_MOD . 'user/background/img/' . $sUsername . PH7_SH . $sFile);
        (new UserCoreModel)->deleteBackground($iProfileId);

        /* Clean User Background Cache */
        (new Cache)->start(UserCoreModel::CACHE_GROUP, 'background' . $iProfileId, null)->clear();
    }

    /**
     * Get the Profile Link.
     *
     * @param string $sUsername
     *
     * @return string The Absolute Profile Link
     */
    public function getProfileLink($sUsername)
    {
        $sUsername = (new Str)->lower($sUsername);
        //return (strlen($sUsername) >1) ? PH7_URL_ROOT . $sUsername . PH7_PAGE_EXT : '#';

        return PH7_URL_ROOT . $sUsername . PH7_PAGE_EXT;
    }

    /**
     * Get Profile Link with the link to the registration form if the user is not connected.
     *
     * @param string $sUsername
     * @param string $sFirstName
     * @param string $sSex
     *
     * @return string The link
     */
    public function getProfileSignupLink($sUsername, $sFirstName, $sSex)
    {
        if (!self::auth() && !AdminCore::auth()) {
            $aHttpParams = [
                'ref' => (new HttpRequest)->currentController(),
                'a' => Registry::getInstance()->action,
                'u' => $sUsername,
                'f_n' => $sFirstName,
                's' => $sSex
            ];

            $sLink = Uri::get('user','signup','step1', '?' . Url::httpBuildQuery($aHttpParams), false);
        }
        else
        {
            $sLink = $this->getProfileLink($sUsername);
        }

       return $sLink;
    }

    /**
     * Set a user authentication.
     *
     * @param stdClass $oUserData User database object.
     * @param UserCoreModel $oUserModel
     * @param Session $oSession
     * @param SecurityModel $oSecurityModel
     *
     * @return void
     */
    public function setAuth(stdClass $oUserData, UserCoreModel $oUserModel, Session $oSession, SecurityModel $oSecurityModel)
    {
        // Remove the session if the user is logged on as "affiliate" or "administrator".
        if (AffiliateCore::auth() || AdminCore::auth())
            $oSession->destroy();

        // Regenerate the session ID to prevent session fixation attack
        $oSession->regenerateId();

        // Now we connect the member
        $aSessionData = [
            'member_id' => $oUserData->profileId,
            'member_email' => $oUserData->email,
            'member_username' => $oUserData->username,
            'member_first_name' => $oUserData->firstName,
            'member_sex' => $oUserData->sex,
            'member_group_id' => $oUserData->groupId,
            'member_ip' => Ip::get(),
            'member_http_user_agent' => (new Browser)->getUserAgent(),
            'member_token' => Various::genRnd($oUserData->email)
        ];

        $oSession->set($aSessionData);

        $oSecurityModel->addLoginLog($oUserData->email, $oUserData->username, '*****', 'Logged in!');
        $oUserModel->setLastActivity($oUserData->profileId);
    }

    /**
     * Finds a free username in our database to use for Facebook connect.
     *
     * @param string $sNickname
     * @param string $sFirstName
     * @param string $sLastName
     *
     * @return string Username
     */
    public function findUsername($sNickname, $sFirstName, $sLastName)
    {
        $iMaxLen = DbConfig::getSetting('maxUsernameLength');
        $sRnd = Various::genRnd('pH_Pierre-Henry_Soria_Sanz_González', 4); // Random String

        $aUsernameList = [
            $sNickname,
            $sFirstName,
            $sLastName,
            $sNickname . $sRnd,
            $sFirstName . $sRnd,
            $sLastName . $sRnd,
            $sFirstName . '-' . $sLastName,
            $sLastName . '-' . $sFirstName,
            $sFirstName . '-' . $sLastName . $sRnd,
            $sLastName . '-' . $sFirstName . $sRnd
        ];

        foreach ($aUsernameList as $sUsername) {
            $sUsername = substr($sUsername, 0, $iMaxLen);

            if ((new Validate)->username($sUsername))
                break;
            else
                $sUsername = Various::genRnd('pOH_Pierre-Henry_Soria_Béghin_Rollier', $iMaxLen); // Default value
        }

        return $sUsername;
    }

    /**
     * Check account status of profile.
     *
     * @param stdClass $oDbProfileData User database object.
     *
     * @return bool|string Returns a boolean TRUE if the account status is correct, otherwise returns an error message.
     */
    public function checkAccountStatus(stdClass $oDbProfileData)
    {
        $mRet = true; // Default value

        if ($oDbProfileData->active != 1) {
            if ($oDbProfileData->active == 2) {
                $mRet = t('Sorry, your account has not been activated yet. Please activate it by clicking the activation link that was emailed.');
            } elseif ($oDbProfileData->active == 3) {
                $mRet = t('Sorry, your account has not been activated yet. An administrator must validate your account.');
            } else {
                $mRet = t('Your account does not have a valid activation status. Please contact the database administrator so that it solves this problem.');
            }
        } elseif ($oDbProfileData->ban == 1) {
            $mRet = t('Sorry, Your account has been banned.');
        }

        return $mRet;
    }

    /**
     * Message and Redirection for Activate Account.
     *
     * @param string $sEmail
     * @param string $sHash
     * @param Config $oConfig
     * @param Registry $oRegistry
     * @param string $sMod (user, affiliate, newsletter).
     *
     * @return void
     */
    public function activateAccount($sEmail, $sHash, Config $oConfig, Registry $oRegistry, $sMod = 'user')
    {
        $sTable = VariousModel::convertModToTable($sMod);
        $sRedirectLoginUrl = ($sMod == 'newsletter' ? PH7_URL_ROOT : ($sMod == 'affiliate' ? Uri::get('affiliate', 'home', 'login') : Uri::get('user', 'main', 'login')));
        $sRedirectIndexUrl = ($sMod == 'newsletter' ? PH7_URL_ROOT : ($sMod == 'affiliate' ? Uri::get('affiliate', 'home', 'index') : Uri::get('user', 'main', 'index')));
        $sSuccessMsg = ($sMod == 'newsletter' ? t('Your subscription to our newsletters has been successfully validated!') : t('Your account has been successfully validated. You can now login!'));

        if (isset($sEmail, $sHash)) {
            $oUserModel = new AffiliateCoreModel;
            if ($oUserModel->validateAccount($sEmail, $sHash, $sTable)) {
                $iId = $oUserModel->getId($sEmail, null, $sTable);
                if ($sMod != 'newsletter')
                    $this->clearReadProfileCache($iId, $sTable);

                /** Update the Affiliate Commission **/
                $iAffId = $oUserModel->getAffiliatedId($iId);
                AffiliateCore::updateJoinCom($iAffId, $oConfig, $oRegistry);

                Header::redirect($sRedirectLoginUrl, $sSuccessMsg);
            }
            else
            {
                Header::redirect($sRedirectLoginUrl, t('Oops! The URL is either invalid or you already have activated your account.'), 'error');
            }
            unset($oUserModel);
        }
        else
        {
            Header::redirect($sRedirectIndexUrl, t('Invalid approach, please use the link that has been send to your email.'), 'error');
        }
    }

    /**
     * Get the correct matching sex.
     *
     * @param string $sSex
     *
     * @return string The Match Sex.
     */
    public function getMatchSex($sSex)
    {
        return ($sSex == 'male' ? 'female' : ($sSex == 'female' ? 'male' : 'couple'));
    }

    /**
     * This method is a wrapper for the cache of the profile of users.
     * Clean UserCoreModel / readProfile Cache
     *
     * @param integer $iId Profile ID.
     * @param string $sTable Default 'Members'
     *
     * @return void
     */
    public function clearReadProfileCache($iId, $sTable = 'Members')
    {
        $this->_clearCache('readProfile', $iId, $sTable);
    }

    /**
     * This method is a wrapper for the Info Fields cache.
     * Clean UserCoreModel / infoFields Cache
     *
     * @param integer $iId Profile ID.
     * @param string $sTable Default 'MembersInfo'
     *
     * @return void
     */
    public function clearInfoFieldCache($iId, $sTable = 'MembersInfo')
    {
        $this->_clearCache('infoFields', $iId, $sTable);
    }

    /**
     * Generic method to clear the user cache.
     *
     * @param string $sId Cache ID.
     * @param integer $iId User ID.
     * @param string $sTable Table name.
     *
     * @return void
     */
    private function _clearCache($sId, $iId, $sTable)
    {
        VariousModel::checkModelTable($sTable);

        (new Cache)->start(UserCoreModel::CACHE_GROUP, $sId . $iId . $sTable, null)->clear();
    }

    /**
     * Clone is set to private to stop cloning.
     */
    private function __clone() {}
}
